#include "NpLookup.h"
#include "SignInSony.h"
#include "ErrorCodesSony.h"

namespace UnityPlugin
{
	NpLookup::NpLookup()
		: m_TitleCtxId(0)
		, m_RequestID(0)
		, m_Result(SCE_OK)
		, m_ResultUserNpID(SCE_OK)
		, m_ResultUserProfile(SCE_OK)
		, m_State(STATE_ALL_FINISHED)
		, m_StateAfterComplete(STATE_ALL_FINISHED)
	{
	}

	NpLookup::~NpLookup()
	{
		Shutdown();
	}

	bool NpLookup::IsBusy() const
	{
		return m_State != STATE_ALL_FINISHED;
	}

	NpLookup::NpLookupResult NpLookup::LookupUserNpID(const SceNpId& myNpID, const SceNpOnlineId& userOnlineID)
	{
		if(m_State != STATE_ALL_FINISHED)
		{
			return NPLOOKUP_BUSY;
		}

		m_State = STATE_START_USER_NPID;
		m_StateAfterComplete = STATE_ALL_FINISHED;
		m_UserOnlineId = userOnlineID;
		memset(&m_UserNpId, 0, sizeof(SceNpId));

		NpLookupResult result = Initialize(myNpID);
		if(result == NPLOOKUP_FAILED)
		{
			printf("Lookup Initialize failed\n");
			return result;
		}

		return DoLookup(false);
	}

	NpLookup::NpLookupResult NpLookup::LookupUserInfo(const SceNpId& myNpID, const SceNpOnlineId& userOnlineID)
	{
		if(m_State != STATE_ALL_FINISHED)
		{
			return NPLOOKUP_BUSY;
		}

		m_UserOnlineId = userOnlineID;
		m_State = STATE_START_USER_NPID;
		m_StateAfterComplete = STATE_START_USER_INFO;
		memset(&m_UserNpId, 0, sizeof(SceNpId));
		memset(&m_UserInfo, 0, sizeof(SceNpUserInfo));
		memset(&m_AboutMe, 0, sizeof(SceNpAboutMe));
		memset(&m_Languages, 0, sizeof(SceNpMyLanguages));
		memset(&m_CountryCode, 0, sizeof(SceNpCountryCode));

		NpLookupResult result = Initialize(myNpID);
		if(result == NPLOOKUP_FAILED)
		{
			printf("Lookup Initialize failed\n");
			return result;
		}

		return DoLookup(false);
	}

	NpLookup::NpLookupResult NpLookup::LookupUserInfo(const SceNpId& myNpID, const SceNpId& userNpID)
	{
		if(m_State != STATE_ALL_FINISHED)
		{
			return NPLOOKUP_BUSY;
		}

		m_UserNpId = userNpID;
		m_State = STATE_START_USER_INFO;
		m_StateAfterComplete = STATE_ALL_FINISHED;
		memset(&m_UserInfo, 0, sizeof(SceNpUserInfo));
		memset(&m_AboutMe, 0, sizeof(SceNpAboutMe));
		memset(&m_Languages, 0, sizeof(SceNpMyLanguages));
		memset(&m_CountryCode, 0, sizeof(SceNpCountryCode));

		NpLookupResult result = Initialize(myNpID);
		if(result == NPLOOKUP_FAILED)
		{
			printf("Lookup Initialize failed\n");
			return result;
		}

		return DoLookup(false);
	}

	NpLookup::NpLookupResult NpLookup::Update()
	{
		int pollingResult = 0;

		if((m_State != STATE_ALL_FINISHED) && !gSignedInState.IsSignedIn())
		{
			return Error(SCE_NP_CORE_ERROR_USER_OFFLINE, __FUNCTION__, __LINE__);
		}

		if(m_RequestID != 0 && m_State != STATE_ALL_FINISHED)
		{
			pollingResult = sceNpLookupPollAsync(m_RequestID, &m_Result);
			if(pollingResult < 0)
			{
				return AsyncOpError(pollingResult, __FUNCTION__, __LINE__);
			}

			if(pollingResult == 0)
			{
				// Finished async operation.
				return DoLookup(true);
			}

			return NPLOOKUP_BUSY;
		}

		return NPLOOKUP_IDLE;
	}

	NpLookup::NpLookupResult NpLookup::DoLookup(bool finishedAsync)
	{
		NpLookupResult returnState = NPLOOKUP_BUSY;
		int ret = SCE_OK;

		if(finishedAsync)
		{
			// Destroy transaction context after finishing async function.
			if(m_RequestID != 0)
			{
				sceNpLookupDestroyTransactionCtx(m_RequestID);
				m_RequestID = 0;
			}

			switch(m_State)
			{
			case STATE_PROCESS_USER_NPID:
				m_ResultUserNpID = m_Result;
				if(m_ResultUserNpID < 0)
				{
					return AsyncOpError(m_ResultUserNpID, __FUNCTION__, __LINE__);
				}
				else
				{
					m_State = m_StateAfterComplete;
					returnState = (m_State == STATE_ALL_FINISHED) ? NPLOOKUP_GOT_NPID : NPLOOKUP_BUSY;
				}
				break;

			case STATE_PROCESS_USER_INFO:
				m_ResultUserProfile = m_Result;
				if(m_ResultUserProfile < 0)
				{
					return AsyncOpError(m_ResultUserProfile, __FUNCTION__, __LINE__);
				}
				else
				{
					m_State = STATE_ALL_FINISHED;
					returnState = NPLOOKUP_GOT_USERINFO;
				}
				break;

			default:
				break;
			}
		}

		// Start Async function.
		switch(m_State)
		{
		case STATE_START_USER_NPID:
			ret = sceNpLookupCreateTransactionCtx(m_TitleCtxId);
			if(ret < 0)
			{
				return AsyncOpError(ret, __FUNCTION__, __LINE__);
			}
			m_RequestID = ret;

			ret = sceNpLookupNpIdAsync(m_RequestID, &m_UserOnlineId, &m_UserNpId, 0, NULL);
			if(ret < 0)
			{
				return AsyncOpError(ret, __FUNCTION__, __LINE__);
			}

			m_State = STATE_PROCESS_USER_NPID;
			break;

		case STATE_START_USER_INFO:
			ret = sceNpLookupCreateTransactionCtx(m_TitleCtxId);
			if(ret < 0)
			{
				return AsyncOpError(ret, __FUNCTION__, __LINE__);
			}
			m_RequestID = ret;

			ret = sceNpLookupUserProfileAsync(
				m_RequestID,
				&m_UserNpId,
				&m_UserInfo,
				&m_AboutMe,
				&m_Languages,
				&m_CountryCode,
				NULL,
				0,
				NULL);
			if(ret < 0)
			{
				return AsyncOpError(ret, __FUNCTION__, __LINE__);
			}
			m_State = STATE_PROCESS_USER_INFO;
			break;

		default:
			break;
		}

		return returnState;
	}

	NpLookup::NpLookupResult NpLookup::AsyncOpError(int errorCode, const char* function, int line)
	{
		if(m_RequestID != 0)
		{
			sceNpLookupDestroyTransactionCtx(m_RequestID);
			m_RequestID = 0;
		}
		char message[512];
		sprintf(message, "NpLookup::%s@L%d - %s", function, line, LookupSceErrorCode(errorCode));
		m_LastErrorCode = errorCode;
		m_LastErrorMessage = message;
		m_State = STATE_ALL_FINISHED;
		return NPLOOKUP_FAILED;
	}

	NpLookup::NpLookupResult NpLookup::Error(int errorCode, const char* function, int line)
	{
		char message[512];
		sprintf(message, "NpLookup::%s@L%d - %s", function, line, LookupSceErrorCode(errorCode));
		m_LastErrorCode = errorCode;
		m_LastErrorMessage = message;
		m_State = STATE_ALL_FINISHED;
		return NPLOOKUP_FAILED;
	}

	NpLookup::NpLookupResult NpLookup::Initialize(const SceNpId& myNpID)
	{
		static bool initialized = false;
		if(initialized)
		{
			return NPLOOKUP_OK;
		}

		int ret = cellSysmoduleIsLoaded(CELL_SYSMODULE_HTTPS);
		if(ret != SCE_OK)
		{
			ret = cellSysmoduleLoadModule(CELL_SYSMODULE_HTTPS);
			if (ret < 0)
			{
				return Error(ret, __FUNCTION__, __LINE__);
			}
		}

		ret = cellSysmoduleIsLoaded(CELL_SYSMODULE_SYSUTIL_NP);
		if(ret != SCE_OK)
		{
			ret = cellSysmoduleLoadModule(CELL_SYSMODULE_SYSUTIL_NP);
			if (ret < 0)
			{
				return Error(ret, __FUNCTION__, __LINE__);
			}
		}

		ret = sceNpLookupInit();

		if(ret < 0 && ret != SCE_NP_COMMUNITY_ERROR_ALREADY_INITIALIZED)
		{
			return Error(ret, __FUNCTION__, __LINE__);
		}
		
		// Create title context
		SceNpCommunicationId* commID = (SceNpCommunicationId*)gAppInfo.m_NpCommunicationId;
		ret = sceNpLookupCreateTitleCtx(commID, &myNpID);
		if (ret < 0)
		{
			return Error(ret, __FUNCTION__, __LINE__);
		}
		m_TitleCtxId = ret;

		initialized = true;
		return NPLOOKUP_OK;
	}

	void NpLookup::Shutdown()
	{
		if(m_RequestID != 0)
		{
			sceNpLookupDestroyTransactionCtx(m_RequestID);
			m_RequestID = 0;
		}

		if(m_TitleCtxId != 0)
		{
			sceNpLookupDestroyTitleCtx(m_TitleCtxId);
			m_TitleCtxId = 0;
		}
		if(m_TitleCtxId != 0)
		{
			sceNpLookupTerm();
		}
	}

};
